import { NextRequest, NextResponse } from "next/server";
import { createAdminClient } from "@/utils/supabase/admin";
import { createClient } from "@/utils/supabase/server";

// 获取单个提交详情
export async function GET(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    // 验证管理员权限
    const adminClient = createAdminClient();
    const supabase = await createClient();
    
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      return NextResponse.json(
        { error: "未授权，请先登录" },
        { status: 401 }
      );
    }
    
    const { data: adminUser } = await adminClient
      .from("admin_users")
      .select("*")
      .eq("user_id", user.id)
      .single();
      
    if (!adminUser) {
      return NextResponse.json(
        { error: "您没有管理员权限" },
        { status: 403 }
      );
    }
    
    // 获取提交详情
    const { data: submission, error } = await adminClient
      .from("tool_submissions")
      .select(`
        *,
        category:categories(id, name)
      `)
      .eq("id", params.id)
      .single();
      
    if (error) {
      console.error("获取提交详情错误:", error);
      return NextResponse.json(
        { error: "获取提交详情失败" },
        { status: 500 }
      );
    }
    
    if (!submission) {
      return NextResponse.json(
        { error: "未找到指定提交" },
        { status: 404 }
      );
    }
    
    // 获取用户信息
    if (submission.user_id) {
      const { data: userData } = await adminClient
        .from("auth.users")
        .select("id, email")
        .eq("id", submission.user_id)
        .single();
        
      if (userData) {
        submission.user = {
          id: userData.id,
          email: userData.email
        };
      }
    }
    
    return NextResponse.json({ submission });
    
  } catch (error) {
    console.error("处理获取提交详情请求错误:", error);
    return NextResponse.json(
      { error: "处理获取提交详情请求失败" },
      { status: 500 }
    );
  }
}

// 通过提交 (批准工具)
export async function POST(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    // 验证管理员权限
    const adminClient = createAdminClient();
    const supabase = await createClient();
    
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      return NextResponse.json(
        { error: "未授权，请先登录" },
        { status: 401 }
      );
    }
    
    const { data: adminUser } = await adminClient
      .from("admin_users")
      .select("*")
      .eq("user_id", user.id)
      .single();
      
    if (!adminUser) {
      return NextResponse.json(
        { error: "您没有管理员权限" },
        { status: 403 }
      );
    }
    
    // 获取提交数据
    const { data: submission, error: getError } = await adminClient
      .from("tool_submissions")
      .select("*")
      .eq("id", params.id)
      .eq("status", "pending")
      .single();
      
    if (getError || !submission) {
      console.error("获取提交数据错误:", getError);
      return NextResponse.json(
        { error: "未找到待审核的提交" },
        { status: 404 }
      );
    }
    
    // 准备AI工具数据
    const now = new Date().toISOString();
    const toolData = {
      name: submission.name,
      slug: submission.slug,
      description: submission.description,
      full_description: submission.full_description,
      website_url: submission.website_url,
      image_url: submission.image_url,
      logo_url: submission.logo_url || null,
      category_id: submission.category_id,
      user_id: submission.user_id,
      created_at: now,
      updated_at: now,
      is_approved: true,
      type: "工具" // 默认类型
    };
    
    // 添加到AI工具表
    const { data: aiTool, error: insertError } = await adminClient
      .from("ai_tools")
      .insert(toolData)
      .select()
      .single();
      
    if (insertError) {
      console.error("添加AI工具错误:", insertError);
      return NextResponse.json(
        { error: "添加AI工具失败" },
        { status: 500 }
      );
    }
    
    // 更新提交状态
    const { error: updateError } = await adminClient
      .from("tool_submissions")
      .update({
        status: "approved",
        processed_at: now,
        ai_tool_id: aiTool.id
      })
      .eq("id", params.id);
      
    if (updateError) {
      console.error("更新提交状态错误:", updateError);
      return NextResponse.json(
        { error: "更新提交状态失败" },
        { status: 500 }
      );
    }
    
    return NextResponse.json({
      message: "提交已通过，AI工具已添加",
      submission_id: params.id,
      tool_id: aiTool.id
    });
    
  } catch (error) {
    console.error("处理通过提交请求错误:", error);
    return NextResponse.json(
      { error: "处理通过提交请求失败" },
      { status: 500 }
    );
  }
}

// 驳回提交
export async function PUT(
  request: NextRequest,
  { params }: { params: { id: string } }
) {
  try {
    // 验证管理员权限
    const adminClient = createAdminClient();
    const supabase = await createClient();
    
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      return NextResponse.json(
        { error: "未授权，请先登录" },
        { status: 401 }
      );
    }
    
    const { data: adminUser } = await adminClient
      .from("admin_users")
      .select("*")
      .eq("user_id", user.id)
      .single();
      
    if (!adminUser) {
      return NextResponse.json(
        { error: "您没有管理员权限" },
        { status: 403 }
      );
    }
    
    // 获取请求体
    const { admin_comments } = await request.json();
    
    // 获取提交数据
    const { data: submission, error: getError } = await adminClient
      .from("tool_submissions")
      .select("*")
      .eq("id", params.id)
      .eq("status", "pending")
      .single();
      
    if (getError || !submission) {
      console.error("获取提交数据错误:", getError);
      return NextResponse.json(
        { error: "未找到待审核的提交" },
        { status: 404 }
      );
    }
    
    // 更新提交状态
    const now = new Date().toISOString();
    const { error: updateError } = await adminClient
      .from("tool_submissions")
      .update({
        status: "rejected",
        processed_at: now,
        admin_comments: admin_comments || "未通过审核"
      })
      .eq("id", params.id);
      
    if (updateError) {
      console.error("更新提交状态错误:", updateError);
      return NextResponse.json(
        { error: "更新提交状态失败" },
        { status: 500 }
      );
    }
    
    return NextResponse.json({
      message: "提交已驳回",
      submission_id: params.id
    });
    
  } catch (error) {
    console.error("处理驳回提交请求错误:", error);
    return NextResponse.json(
      { error: "处理驳回提交请求失败" },
      { status: 500 }
    );
  }
} 